%    Implementing Finite difference method for the pricing of a simple contingent claim %
%    Based on: Lamberton, Lapeyre 
%    "Introduction to Stochastic Calculus applied to Finance", 1996, Ch 5.%
%    Created by: Nik Tuzov, www.ntuzov.com
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%    function value = claim(contract, euro_amer, S, r, sigma, T, L, N, M, theta)      
%
%                          Input parameters:
%       contract    - contract function for a simple claim, should be
%                     defined in a separate *.m file
%       euro_amer   - is equal to 'euro' for a European contract and 'amer'
%                     for an American contract%
%      S            - underlying spot price, $
%      r            -  risk - free rate (0.05 means 5%)
%      sigma        -  volatility per unit of time ( 0.3 means 30%)
%      T            -  time to maturity
%      L            -  state space localization parameter: the log(S)
%                      belongs to [-L; L]
%      N            -  state space discretization: localized state space 
%                      [-L; L] has N+2 nodes
%      M            -  time axis discretization: [0; T] has M+1 nodes    
%      theta        -  theta for theta-schemes, ranges from 0 to 1:  
%                      theta = 1: fully implicit method
%                      theta = 0: fully explicit method
%                      theta = 0.5: Crank and Nicholson scheme 
% 
%                           Returned value:
%     A real number - price of the specified contract. It can be positive
%     or negative depending on the contract function.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 

     function value = claim(contract, euro_amer, S, r, sigma, T, L, N, M, theta)

     k = T / M;                     % time step
     h = 2 * L / (N + 1);           % space step          
     X = -L : h : L;                % state space grid, N+2 points
     X = X(2 : N + 1);              % remove f_0 and f_(N+1) from the vector     
     
     % evaluating the contract function on the space grid:     
     f_h = contract(exp(X));      
     
     alpha = sigma^2 / (2 * h^2) - (1 / (2*h)) * (r - sigma^2 / 2);
     beta = -sigma^2 / h^2 - r;
     gamma = sigma^2 / (2 * h^2) + (1 / (2*h)) * (r - sigma^2 / 2);
     
     % a, b, c: (p 109 and 115):
     a = -k * theta * alpha;
     b = -k * theta * beta + 1;
     c = -k * theta * gamma;
     
    % d1, d2, d3 needed to simplify computation of G:    
     d2 = 1 + (1 - theta) * k * beta;
     d1 = (1 - theta) * k * alpha;
     d3 = (1 - theta) * k * gamma;     
 
     % Solve backwards w.r.t time from T to zero:  
     U_last = f_h';
     U_next = U_last;     
     G = zeros(N, 1);
     
      for n = (M - 1) : -1 : 0                       
         
         if (euro_amer == 'euro')
          G(1) = d2 * U_last(1) + d3 * U_last(2);
         else
           G(1) = (1 + k * (1 - theta) * (beta + alpha)) * U_last(1) + k * (1-theta) * gamma * U_last(2);  
         end;
         
         for i = 2 : (N - 1)
             G(i) = d1 * U_last(i-1) + d2 * U_last(i) + d3 * U_last(i+1);
         end;
         
         if (euro_amer == 'euro')
           G(N) = d1 * U_last(N-1) + d2 * U_last(N); 
         else
            G(N) = k * (1 - theta) * alpha * U_last(N-1) + (1 + k * (1 - theta) * (beta + gamma) ) * U_last(N);
         end;
        
         % Gauss method begins:
         g_prime = zeros(N,1);
         b_prime = zeros(N,1);            
         
         if (euro_amer == 'euro')
           b_prime(N) = b;
         else
            b_prime(N) = b + c;
         end;         
         g_prime(N) = G(N);
         
         if (euro_amer == 'euro')  % European
             for i = (N-1) : -1 : 1
                 b_prime(i) = b - c * a / b_prime(i+1);
                 g_prime(i) = G(i) - c * g_prime(i+1) / b_prime(i+1);
             end;
         else                    % American
               for i = (N-1) : -1 : 1
                 if (i == 1) 
                   b_i = a + b;
                 else
                   b_i = b;
                 end;
               b_prime(i) = b_i - c * a / b_prime(i+1);
               g_prime(i) = G(i) - c * g_prime(i+1) / b_prime(i+1);
               end; % for i        
         end;         
         
         U_next(1) = g_prime(1) / b_prime(1); 
         
         for i = 2 : N
             U_next(i) = (g_prime(i) - a * U_next(i-1)) / b_prime(i);
             
             if (euro_amer == 'amer')   % American 
                 U_next(i) = max(U_next(i), f_h(i));
             end;
         end;
         % Gauss method ends;         
                     
         U_last = U_next;         
     end;  % for n
     
     % finding the final answer:     
     [min_diff, index] = min( abs( X - log(S)) );      
     value = U_next(index);               
     
% End of function